<?php
// regional_analysis.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;
use Carbon\Carbon;

// دریافت پارامترهای فیلتر
$selected_shamsi_year = $_GET['shamsi_year'] ?? null;
$time_period = $_GET['time_period'] ?? 'yearly';
$selected_city = $_GET['city'] ?? null;
$selected_product = $_GET['product'] ?? null;
$selected_usage_section = $_GET['usage_section'] ?? null;
$show_customers = $_GET['show_customers'] ?? null;
$category_type = $_GET['category_type'] ?? null;
$category_value = $_GET['category_value'] ?? null;
  $back_url = "?";
                        if ($selected_shamsi_year)
                            $back_url .= "shamsi_year=" . urlencode($selected_shamsi_year) . "&";
                        if ($time_period)
                            $back_url .= "time_period=" . urlencode($time_period) . "&";
                        if ($selected_city)
                            $back_url .= "city=" . urlencode($selected_city) . "&";
                        if ($selected_product)
                            $back_url .= "product=" . urlencode($selected_product) . "&";
                        if ($selected_usage_section)
                            $back_url .= "usage_section=" . urlencode($selected_usage_section) . "&";
                        if ($category_type)
                            $back_url .= "category_type=" . urlencode($category_type) . "&";
                        if ($category_value)
                            $back_url .= "category_value=" . urlencode($category_value);

                        // حذف & اضافی از انتهای URL
                        $back_url = rtrim($back_url, "&");
// آرایه مپینگ ستون‌ها
$column_mapping = [
    'بخش مصرف' => 'بخش_مصرف',
    'متولی اول' => 'متولی_اول',
    'گره کاربرد' => 'گره_کاربرد',
    'کاربرد' => 'کاربرد',
    'عنوان کاربرد کاربر' => 'عنوان_کاربرد_کاربر',
    'نام متقاضی' => 'نام_متقاضی',
    'کدملی/ شناسه ملی' => 'کدملی_شناسه_ملی',
    'تلفن همراه' => 'تلفن_همراه',
    'آدرس' => 'آدرس',
    'کد فروش' => 'کد_فروش',
    'استان' => 'استان',
    'شهرستان' => 'شهرستان',
    'شناسه یکتای کاربرد' => 'شناسه_یکتای_کاربرد',
     'شماره درخواست' => 'شماره_درخواست',
    'فراوده' => 'فراوده',
    'نوع تحویل سوخت' => 'نوع_تحویل_سوخت',
    'مقدار تایید شده توسط پخش' => 'مقدار_تایید_شده_توسط_پخش',
    'نام فروشندگی انتخاب شده در درخواست' => 'نام_فروشندگی_انتخاب_شده_در_درخواست',
    'کد فروشندگی انتخاب شده در درخواست' => 'کد_فروشندگی_انتخاب_شده_در_درخواست',
    'تاریخ تایید درخواست توسط پخش' => 'تاریخ_تایید_درخواست_توسط_پخش',
    'نوع درخواست' => 'نوع_درخواست'
];

// توابع تبدیل تاریخ شمسی
function getGregorianRangeFromShamsi($shamsiYear, $period = 'yearly', $period_number = null) {
    try {
        $start_shamsi = null;
        $end_shamsi = null;

        switch($period) {
            case 'monthly':
                if ($period_number && $period_number >= 1 && $period_number <= 12) {
                    $start_shamsi = new Jalalian($shamsiYear, $period_number, 1);
                    $end_shamsi = new Jalalian($shamsiYear, $period_number, 29);
                    try {
                        $test_day = new Jalalian($shamsiYear, $period_number, 30);
                        $end_shamsi = $test_day;
                    } catch (Exception $e) {}
                }
                break;

            case 'quarterly':
                if ($period_number && $period_number >= 1 && $period_number <= 4) {
                    $start_month = (($period_number - 1) * 3) + 1;
                    $end_month = $start_month + 2;
                    $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                    $end_shamsi = new Jalalian($shamsiYear, $end_month, 29);
                    try {
                        $test_day = new Jalalian($shamsiYear, $end_month, 30);
                        $end_shamsi = $test_day;
                    } catch (Exception $e) {}
                }
                break;

            case 'halfyearly':
                if ($period_number && $period_number >= 1 && $period_number <= 2) {
                    $start_month = $period_number == 1 ? 1 : 7;
                    $end_month = $period_number == 1 ? 6 : 12;
                    $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                    $end_shamsi = new Jalalian($shamsiYear, $end_month, 29);
                    try {
                        $test_day = new Jalalian($shamsiYear, $end_month, 30);
                        $end_shamsi = $test_day;
                    } catch (Exception $e) {}
                }
                break;

            case 'yearly':
            default:
                $start_shamsi = new Jalalian($shamsiYear, 1, 1);
                $end_shamsi = new Jalalian($shamsiYear, 12, 29);
                try {
                    $test_day = new Jalalian($shamsiYear, 12, 30);
                    $end_shamsi = $test_day;
                } catch (Exception $e) {}
                break;
        }

        if ($start_shamsi && $end_shamsi) {
            $start_gregorian = $start_shamsi->toCarbon();
            $end_gregorian = $end_shamsi->toCarbon();
            $end_gregorian->setTime(23, 59, 59);
            
            return [
                'start' => $start_gregorian->format('Y-m-d H:i:s'),
                'end' => $end_gregorian->format('Y-m-d H:i:s'),
                'shamsi_period' => getShamsiPeriodName($period, $period_number, $shamsiYear)
            ];
        }
    } catch (Exception $e) {
        return null;
    }
    return null;
}

function getShamsiPeriodName($period, $period_number, $year) {
    $month_names = [
        1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد', 4 => 'تیر',
        5 => 'مرداد', 6 => 'شهریور', 7 => 'مهر', 8 => 'آبان',
        9 => 'آذر', 10 => 'دی', 11 => 'بهمن', 12 => 'اسفند'
    ];
    
    $quarter_names = [1 => 'اول', 2 => 'دوم', 3 => 'سوم', 4 => 'چهارم'];
    $halfyear_names = [1 => 'اول', 2 => 'دوم'];
    
    switch($period) {
        case 'monthly':
            return $month_names[$period_number] . ' ' . $year;
        case 'quarterly':
            return 'فصل ' . $quarter_names[$period_number] . ' ' . $year;
        case 'halfyearly':
            return 'نیمه ' . $halfyear_names[$period_number] . ' ' . $year;
        case 'yearly':
            return 'سال ' . $year;
        default:
            return 'سال ' . $year;
    }
}

function getAvailableShamsiYears($pdo) {
    $query = $pdo->query("
        SELECT DISTINCT تاریخ_تایید_درخواست_توسط_پخش 
        FROM fuel_requests 
        WHERE تاریخ_تایید_درخواست_توسط_پخش IS NOT NULL 
        ORDER BY تاریخ_تایید_درخواست_توسط_پخش DESC
    ");
    $dates = $query->fetchAll(PDO::FETCH_COLUMN);

    $years = [];
    foreach ($dates as $date) {
        try {
            $carbon = Carbon::parse($date);
            $jalali = Jalalian::fromCarbon($carbon);
            $shamsi_year = $jalali->getYear();
            if (!in_array($shamsi_year, $years)) {
                $years[] = $shamsi_year;
            }
        } catch (Exception $e) {
            continue;
        }
    }

    rsort($years);
    return $years;
}

// دریافت داده‌های فیلتر
$available_shamsi_years = getAvailableShamsiYears($pdo);
$cities = $pdo->query("SELECT DISTINCT شهرستان FROM fuel_requests WHERE شهرستان IS NOT NULL ORDER BY شهرستان")->fetchAll(PDO::FETCH_COLUMN);
$products = $pdo->query("SELECT DISTINCT فراوده FROM fuel_requests WHERE فراوده IS NOT NULL ORDER BY فراوده")->fetchAll(PDO::FETCH_COLUMN);
$usage_sections = $pdo->query("SELECT DISTINCT بخش_مصرف FROM fuel_requests WHERE بخش_مصرف IS NOT NULL ORDER BY بخش_مصرف")->fetchAll(PDO::FETCH_COLUMN);

// محاسبه دوره‌های زمانی
$periods_data = [];
if ($selected_shamsi_year) {
    switch($time_period) {
        case 'monthly':
            for ($i = 1; $i <= 12; $i++) {
                $range = getGregorianRangeFromShamsi($selected_shamsi_year, 'monthly', $i);
                if ($range) {
                    $periods_data[] = $range;
                }
            }
            break;
        case 'quarterly':
            for ($i = 1; $i <= 4; $i++) {
                $range = getGregorianRangeFromShamsi($selected_shamsi_year, 'quarterly', $i);
                if ($range) {
                    $periods_data[] = $range;
                }
            }
            break;
        case 'halfyearly':
            for ($i = 1; $i <= 2; $i++) {
                $range = getGregorianRangeFromShamsi($selected_shamsi_year, 'halfyearly', $i);
                if ($range) {
                    $periods_data[] = $range;
                }
            }
            break;
        case 'yearly':
        default:
            $range = getGregorianRangeFromShamsi($selected_shamsi_year, 'yearly');
            if ($range) {
                $periods_data[] = $range;
            }
            break;
    }
}

// ساخت شرط‌های WHERE و دریافت داده‌ها
$analysis_data = [];
foreach ($periods_data as $period_index => $period_range) {
    $where_conditions = [];
    $params = [];
    
    // شرط زمانی
    $where_conditions[] = "تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
    $params[] = $period_range['start'];
    $params[] = $period_range['end'];
    
    // فیلترهای دیگر
    if ($selected_city) {
        $where_conditions[] = "شهرستان = ?";
        $params[] = $selected_city;
    }
    
    if ($selected_product) {
        $where_conditions[] = "فراوده = ?";
        $params[] = $selected_product;
    }
    
    if ($selected_usage_section) {
        $where_conditions[] = "بخش_مصرف = ?";
        $params[] = $selected_usage_section;
    }
    
    $where_sql = "WHERE " . implode(" AND ", $where_conditions);
    
    // دریافت داده‌ها برای این دوره
    $period_data = [];
    
    // 1. میزان مصرف کل ناحیه
    $total_customers = $pdo->prepare("
        SELECT SUM(مقدار_تایید_شده_توسط_پخش) as total_customers
        FROM fuel_requests 
        $where_sql
    ");
    $total_customers->execute($params);
    $period_data['total_customers'] = $total_customers->fetch(PDO::FETCH_ASSOC)['total_customers'];

    // 2. نمودار میزان مصرف ناحیه - شهرستان
    $region_city = $pdo->prepare("
        SELECT 
            استان as region,
            شهرستان as city,
            SUM(مقدار_تایید_شده_توسط_پخش) as customer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY استان, شهرستان
        ORDER BY customer_count DESC
    ");
    $region_city->execute($params);
    $period_data['region_city'] = $region_city->fetchAll(PDO::FETCH_ASSOC);

    // 3.نمودار میزان مصرف- ناحیه (شهرستان)
    $city_customers = $pdo->prepare("
        SELECT 
            شهرستان as city,
            SUM(مقدار_تایید_شده_توسط_پخش) as customer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY شهرستان
        ORDER BY customer_count DESC
    ");
    $city_customers->execute($params);
    $period_data['city_customers'] = $city_customers->fetchAll(PDO::FETCH_ASSOC);

    // 4. نمودار شهرستان‌ها - فراورده (مقدار تایید شده)
    $city_product_where = array_merge($where_conditions, [
        "مقدار_تایید_شده_توسط_پخش IS NOT NULL",
        "مقدار_تایید_شده_توسط_پخش > 0"
    ]);
    $city_product_sql = "WHERE " . implode(" AND ", $city_product_where);
    
    $city_product = $pdo->prepare("
        SELECT 
            شهرستان as city,
            فراوده as product,
            SUM(مقدار_تایید_شده_توسط_پخش) as total_quantity,
            SUM(مقدار_تایید_شده_توسط_پخش) as customer_count
        FROM fuel_requests 
        $city_product_sql
        GROUP BY شهرستان, فراوده
        ORDER BY شهرستان, total_quantity DESC
    ");
    $city_product->execute($params);
    $period_data['city_product'] = $city_product->fetchAll(PDO::FETCH_ASSOC);

    // 5. نمودار ناحیه - کاربری (بخش مصرف)
    $region_usage = $pdo->prepare("
        SELECT 
            استان as region,
            بخش_مصرف as usage_section,
            SUM(مقدار_تایید_شده_توسط_پخش) as customer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY استان, بخش_مصرف
        ORDER BY استان, customer_count DESC
    ");
    $region_usage->execute($params);
    $period_data['region_usage'] = $region_usage->fetchAll(PDO::FETCH_ASSOC);

    // 6. نمودار شهرستان - کاربری (بخش مصرف)
    $city_usage = $pdo->prepare("
        SELECT 
            شهرستان as city,
            بخش_مصرف as usage_section,
            SUM(مقدار_تایید_شده_توسط_پخش) as customer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY شهرستان, بخش_مصرف
        ORDER BY شهرستان, customer_count DESC
    ");
    $city_usage->execute($params);
    $period_data['city_usage'] = $city_usage->fetchAll(PDO::FETCH_ASSOC);

    // 7. نمودار کاربری (بخش مصرف) - فرآورده
    $usage_product = $pdo->prepare("
        SELECT 
            بخش_مصرف as usage_section,
            فراوده as product,
            SUM(مقدار_تایید_شده_توسط_پخش) as customer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY بخش_مصرف, فراوده
        ORDER BY بخش_مصرف, customer_count DESC
    ");
    $usage_product->execute($params);
    $period_data['usage_product'] = $usage_product->fetchAll(PDO::FETCH_ASSOC);
    
    // ذخیره داده‌های این دوره
    $analysis_data[$period_range['shamsi_period']] = $period_data;
}

?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تحلیل منطقه‌ای مصرف‌کنندگان</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        @font-face {
            font-family: "Iran Sans";
            src: url('https://central.travianlegacy.com/fonts/iransansweb.ttf') format('truetype'), url('https://central.travianlegacy.com/fonts/iransansweb.woff') format('woff');
            font-weight: normal;
            font-style: normal;
        }

        body {
            font-family: 'Iran Sans' !important;
            background-color: #f8f9fa;
        }

        h1,
        h2,
        h3 {
            font-family: 'B Titr', Arial !important;
        }

        p,
        div,
        span {
            font-family: 'B Nazanin', Tahoma !important;
        }

        .year-section {
            margin-bottom: 20px;
        }

        .year-header {
            cursor: pointer;
            padding: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 10px;
            margin-bottom: 15px;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }

        .year-header:hover {
            background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
            transform: translateY(-2px);
        }

        .year-header h4 {
            margin: 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .analysis-section {
            display: none;
            margin-bottom: 15px;
        }

        .analysis-section.show {
            display: block;
        }

        .analysis-header {
            cursor: pointer;
            padding: 15px;
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            color: white;
            border-radius: 8px;
            margin-bottom: 10px;
            transition: all 0.3s ease;
        }

        .analysis-header:hover {
            background: linear-gradient(135deg, #f5576c 0%, #f093fb 100%);
        }

        .analysis-content {
            display: none;
            padding: 20px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            margin-bottom: 15px;
        }

        .analysis-content.show {
            display: block;
        }

        .toggle-icon {
            transition: transform 0.3s ease;
        }

        .toggle-icon.rotated {
            transform: rotate(180deg);
        }

        .chart-container {
            position: relative;
            margin: 20px 0;
            height: 400px;
        }

        .dataset-table {
            margin-top: 20px;
            font-size: 0.9em;
        }

        .control-buttons {
            margin-bottom: 20px;
        }

        .stat-card {
            background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
            color: white;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            margin-bottom: 20px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }

        .stat-card h3 {
            margin: 0;
            font-size: 2.5rem;
            font-weight: bold;
        }

        .stat-card p {
            margin: 10px 0 0 0;
            font-size: 1.1rem;
        }

        .customer-list-btn {
            margin: 10px 0;
        }

        .percentage-badge {
            background: #28a745;
            color: white;
            padding: 3px 8px;
            border-radius: 12px;
            font-size: 0.8em;
            margin-right: 5px;
        }
    </style>
</head>

<body>
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-12">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">تحلیل منطقه‌ای مصرف‌کنندگان</h4>
                    </div>
                    <div class="card-body">
                       <div class="card mb-4">
    <div class="card-header bg-info text-white">
        <h5 class="mb-0">فیلترهای تحلیل</h5>
    </div>
    <div class="card-body">
        <form method="GET" class="row g-3">
            <!-- فیلتر زمانی -->
            <div class="col-md-3">
                <label class="form-label">سال شمسی</label>
                <select class="form-select" name="shamsi_year">
                    <option value="">انتخاب سال</option>
                    <?php foreach($available_shamsi_years as $year): ?>
                        <option value="<?= $year ?>" <?= $selected_shamsi_year == $year ? 'selected' : '' ?>>
                            <?= $year ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="col-md-3">
                <label class="form-label">دوره زمانی</label>
                <select class="form-select" name="time_period">
                    <option value="yearly" <?= $time_period == 'yearly' ? 'selected' : '' ?>>سالانه</option>
                    <option value="halfyearly" <?= $time_period == 'halfyearly' ? 'selected' : '' ?>>شش ماهه</option>
                    <option value="quarterly" <?= $time_period == 'quarterly' ? 'selected' : '' ?>>فصلی</option>
                    <option value="monthly" <?= $time_period == 'monthly' ? 'selected' : '' ?>>ماهانه</option>
                </select>
            </div>

            <!-- فیلترهای قابل انتخاب -->
            <div class="col-md-3">
                <label class="form-label">شهرستان</label>
                <select class="form-select" name="city">
                    <option value="">همه شهرستان‌ها</option>
                    <?php foreach($cities as $city): ?>
                        <option value="<?= $city ?>" <?= $selected_city == $city ? 'selected' : '' ?>><?= $city ?></option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="col-md-3">
                <label class="form-label">فراورده</label>
                <select class="form-select" name="product">
                    <option value="">همه فراورده‌ها</option>
                    <?php foreach($products as $product): ?>
                        <option value="<?= $product ?>" <?= $selected_product == $product ? 'selected' : '' ?>><?= $product ?></option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="col-md-3">
                <label class="form-label">بخش مصرف</label>
                <select class="form-select" name="usage_section">
                    <option value="">همه بخش‌ها</option>
                    <?php foreach($usage_sections as $usage): ?>
                        <option value="<?= $usage ?>" <?= $selected_usage_section == $usage ? 'selected' : '' ?>><?= $usage ?></option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="col-12">
                <button type="submit" class="btn btn-primary">اعمال فیلترها</button>
                <a href="regional_analysis.php" class="btn btn-secondary">پاک کردن همه</a>
            </div>
        </form>
    </div>
</div>


                        <?php if ($selected_shamsi_year): ?>
                            <!-- نمایش محدوده تاریخ -->
                          



                            <!-- دکمه‌های کنترل -->
                            <div class="control-buttons">
                                <button class="btn btn-success me-2" onclick="expandAll()">
                                    <i class="bi bi-arrows-expand"></i> باز کردن همه
                                </button>
                                <button class="btn btn-warning" onclick="collapseAll()">
                                    <i class="bi bi-arrows-collapse"></i> بستن همه
                                </button>
                            </div>

                           

                            
                           <!-- نمایش داده‌ها برای هر دوره -->
<?php if (!empty($analysis_data)): ?>
    <?php foreach($analysis_data as $period_name => $period_analysis): ?>
        <div class="year-section">
            <div class="year-header" onclick="toggleYear('<?= $period_name ?>')">
                <h4>
                    <span>تحلیل‌های <?= $period_name ?></span>
                    <i class="bi bi-chevron-down toggle-icon" id="<?= $period_name ?>-icon"></i>
                </h4>
            </div>
            
         <div class="analysis-section show" id="<?= $period_name ?>">
    <!-- 1. میزان مصرف کل ناحیه -->
    <div class="analysis-header" onclick="toggleAnalysis('total-customers-<?= $period_name ?>')">
        <h5>
            <span>۱. میزان مصرف کل ناحیه</span>
            <i class="bi bi-chevron-down toggle-icon" id="total-customers-<?= $period_name ?>-icon"></i>
        </h5>
    </div>
    <div class="analysis-content" id="total-customers-<?= $period_name ?>">
        <div class="alert alert-info">
            <h6>کل تایید بخش: <?= isset($period_analysis['total_customers']) ? number_format($period_analysis['total_customers']) : 0 ?></h6>
            <p class="mb-0">بر اساس شمارش شناسه‌های یکتای کاربرد در <?= $period_name ?></p>
        </div>
    </div>

    <!-- 2. نمودار میزان مصرف ناحیه - شهرستان -->
    <div class="analysis-header" onclick="toggleAnalysis('region-city-<?= $period_name ?>')">
        <h5>
            <span>۲. نمودار میزان مصرف ناحیه - شهرستان</span>
            <i class="bi bi-chevron-down toggle-icon" id="region-city-<?= $period_name ?>-icon"></i>
        </h5>
    </div>
    <div class="analysis-content" id="region-city-<?= $period_name ?>">
        <div class="chart-container">
            <canvas id="regionCityChart-<?= str_replace(' ', '-', $period_name) ?>"></canvas>
        </div>
        <div class="dataset-table">
            <div class="table-responsive">
                <table class="table table-bordered table-striped">
                    <thead class="table-dark">
                        <tr>
                            <th>استان</th>
                            <th>شهرستان</th>
                            <th>میزان مصرف</th>
                            <th>درصد</th>
                            <th>عملیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (isset($period_analysis['region_city']) && is_array($period_analysis['region_city'])): ?>
                            <?php
                            $total = array_sum(array_column($period_analysis['region_city'], 'customer_count'));
                            foreach ($period_analysis['region_city'] as $item):
                                if (isset($item['customer_count']) && isset($item['region']) && isset($item['city'])):
                                    $percentage = $total > 0 ? round(($item['customer_count'] / $total) * 100, 2) : 0;
                            ?>
                                <tr>
                                    <td><?= htmlspecialchars($item['region']) ?></td>
                                    <td><?= htmlspecialchars($item['city']) ?></td>
                                    <td><?= number_format($item['customer_count']) ?></td>
                                    <td>
                                        <span class="percentage-badge"><?= $percentage ?>%</span>
                                    </td>
                                    <td>
                                        <a href="<?= $back_url ?>&show_customers=1&category_type=city&category_value=<?= urlencode($item['city']) ?>#customer-list"
                                            class="btn btn-sm btn-outline-primary">
                                            نمایش لیست مشتریان
                                        </a>
                                    </td>
                                </tr>
                            <?php endif; endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="5" class="text-center text-muted">داده‌ای برای نمایش وجود ندارد</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- 3.نمودار میزان مصرف- ناحیه (شهرستان) -->
    <div class="analysis-header" onclick="toggleAnalysis('city-customers-<?= $period_name ?>')">
        <h5>
            <span>۳.نمودار میزان مصرف- ناحیه (شهرستان)</span>
            <i class="bi bi-chevron-down toggle-icon" id="city-customers-<?= $period_name ?>-icon"></i>
        </h5>
    </div>
    <div class="analysis-content" id="city-customers-<?= $period_name ?>">
        <div class="chart-container">
            <canvas id="cityCustomersChart-<?= str_replace(' ', '-', $period_name) ?>"></canvas>
        </div>
        <div class="dataset-table">
            <div class="table-responsive">
                <table class="table table-bordered table-striped">
                    <thead class="table-dark">
                        <tr>
                            <th>شهرستان</th>
                            <th>میزان مصرف</th>
                            <th>درصد</th>
                            <th>عملیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (isset($period_analysis['city_customers']) && is_array($period_analysis['city_customers'])): ?>
                            <?php
                            $total = array_sum(array_column($period_analysis['city_customers'], 'customer_count'));
                            foreach ($period_analysis['city_customers'] as $item):
                                if (isset($item['customer_count']) && isset($item['city'])):
                                    $percentage = $total > 0 ? round(($item['customer_count'] / $total) * 100, 2) : 0;
                            ?>
                                <tr>
                                    <td><?= htmlspecialchars($item['city']) ?></td>
                                    <td><?= number_format($item['customer_count']) ?></td>
                                    <td>
                                        <span class="percentage-badge"><?= $percentage ?>%</span>
                                    </td>
                                    <td>
                                        <a href="<?= $back_url ?>&show_customers=1&category_type=city&category_value=<?= urlencode($item['city']) ?>#customer-list"
                                            class="btn btn-sm btn-outline-primary">
                                            نمایش لیست مشتریان
                                        </a>
                                    </td>
                                </tr>
                            <?php endif; endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="4" class="text-center text-muted">داده‌ای برای نمایش وجود ندارد</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- 4. نمودار شهرستان‌ها - فراورده (مقدار تایید شده) -->
    <div class="analysis-header" onclick="toggleAnalysis('city-product-<?= $period_name ?>')">
        <h5>
            <span>۴. نمودار شهرستان‌ها - فراورده (مقدار تایید شده)</span>
            <i class="bi bi-chevron-down toggle-icon" id="city-product-<?= $period_name ?>-icon"></i>
        </h5>
    </div>
    <div class="analysis-content" id="city-product-<?= $period_name ?>">
        <div class="alert alert-info">
            <i class="bi bi-info-circle"></i>
            در این بخش می‌توانید توزیع فراورده‌های تایید شده را برای هر شهرستان به صورت جداگانه مشاهده کنید.
        </div>

        <div class="chart-container">
            <canvas id="cityProductChart-<?= str_replace(' ', '-', $period_name) ?>"></canvas>
        </div>

        <div class="dataset-table">
            <div class="table-responsive">
                <table class="table table-bordered table-striped">
                    <thead class="table-dark">
                        <tr>
                            <th>شهرستان</th>
                            <th>فراورده</th>
                            <th>مقدار تایید شده (لیتر)</th>
                            <th>میزان مصرف</th>
                            <th>میانگین به ازای هر مشتری</th>
                            <th>درصد از کل شهرستان</th>
                            <th>عملیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (isset($period_analysis['city_product']) && is_array($period_analysis['city_product'])): ?>
                            <?php
                            // محاسبه مجموع برای هر شهرستان
                            $city_totals = [];
                            foreach ($period_analysis['city_product'] as $item):
                                if (isset($item['city']) && isset($item['total_quantity'])):
                                    if (!isset($city_totals[$item['city']])) {
                                        $city_totals[$item['city']] = 0;
                                    }
                                    $quantity = $item['total_quantity'] ?? 0;
                                    $city_totals[$item['city']] += $quantity;
                                endif;
                            endforeach;

                            foreach ($period_analysis['city_product'] as $item):
                                if (isset($item['city']) && isset($item['product'])):
                                    // بررسی وجود فیلدها
                                    $quantity = $item['total_quantity'] ?? 0;
                                    $customer_count = $item['customer_count'] ?? 0;
                                    
                                    $percentage = isset($city_totals[$item['city']]) && $city_totals[$item['city']] > 0 ?
                                        round(($quantity / $city_totals[$item['city']]) * 100, 2) : 0;
                                    $avg_per_customer = $customer_count > 0 ?
                                        round($quantity / $customer_count, 2) : 0;
                            ?>
                                <tr>
                                    <td><strong><?= htmlspecialchars($item['city']) ?></strong></td>
                                    <td><?= htmlspecialchars($item['product']) ?></td>
                                    <td><?= number_format($quantity, 2) ?></td>
                                    <td><?= number_format($customer_count) ?></td>
                                    <td><?= number_format($avg_per_customer, 2) ?> لیتر</td>
                                    <td>
                                        <span class="badge bg-success"><?= $percentage ?>%</span>
                                    </td>
                                    <td>
                                        <a href="<?= $back_url ?>&show_customers=1&category_type=product&category_value=<?= urlencode($item['product']) ?>&city=<?= urlencode($item['city']) ?>#customer-list"
                                            class="btn btn-sm btn-outline-primary">
                                            <i class="bi bi-people"></i> نمایش مشتریان
                                        </a>
                                    </td>
                                </tr>
                            <?php endif; endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="7" class="text-center text-muted">داده‌ای برای نمایش وجود ندارد</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- 5. نمودار ناحیه - کاربری -->
    <div class="analysis-header" onclick="toggleAnalysis('region-usage-<?= $period_name ?>')">
        <h5>
            <span>۵. نمودار ناحیه - کاربری (بخش مصرف)</span>
            <i class="bi bi-chevron-down toggle-icon" id="region-usage-<?= $period_name ?>-icon"></i>
        </h5>
    </div>
    <div class="analysis-content" id="region-usage-<?= $period_name ?>">
        <div class="chart-container">
            <canvas id="regionUsageChart-<?= str_replace(' ', '-', $period_name) ?>"></canvas>
        </div>
        <div class="dataset-table">
            <div class="table-responsive">
                <table class="table table-bordered table-striped">
                    <thead class="table-dark">
                        <tr>
                            <th>استان</th>
                            <th>بخش مصرف</th>
                            <th>میزان مصرف</th>
                            <th>درصد</th>
                            <th>عملیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (isset($period_analysis['region_usage']) && is_array($period_analysis['region_usage'])): ?>
                            <?php
                            $total = array_sum(array_column($period_analysis['region_usage'], 'customer_count'));
                            foreach ($period_analysis['region_usage'] as $item):
                                if (isset($item['customer_count']) && isset($item['region']) && isset($item['usage_section'])):
                                    $percentage = $total > 0 ? round(($item['customer_count'] / $total) * 100, 2) : 0;
                            ?>
                                <tr>
                                    <td><?= htmlspecialchars($item['region']) ?></td>
                                    <td><?= htmlspecialchars($item['usage_section']) ?></td>
                                    <td><?= number_format($item['customer_count']) ?></td>
                                    <td>
                                        <span class="percentage-badge"><?= $percentage ?>%</span>
                                    </td>
                                    <td>
                                        <a href="<?= $back_url ?>&show_customers=1&category_type=usage_section&category_value=<?= urlencode($item['usage_section']) ?>#customer-list"
                                            class="btn btn-sm btn-outline-primary">
                                            نمایش لیست مشتریان
                                        </a>
                                    </td>
                                </tr>
                            <?php endif; endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="5" class="text-center text-muted">داده‌ای برای نمایش وجود ندارد</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- 6. نمودار شهرستان - کاربری -->
    <div class="analysis-header" onclick="toggleAnalysis('city-usage-<?= $period_name ?>')">
        <h5>
            <span>۶. نمودار شهرستان - کاربری (بخش مصرف)</span>
            <i class="bi bi-chevron-down toggle-icon" id="city-usage-<?= $period_name ?>-icon"></i>
        </h5>
    </div>
    <div class="analysis-content" id="city-usage-<?= $period_name ?>">
        <div class="chart-container">
            <canvas id="cityUsageChart-<?= str_replace(' ', '-', $period_name) ?>"></canvas>
        </div>
        <div class="dataset-table">
            <div class="table-responsive">
                <table class="table table-bordered table-striped">
                    <thead class="table-dark">
                        <tr>
                            <th>شهرستان</th>
                            <th>بخش مصرف</th>
                            <th>میزان مصرف</th>
                            <th>عملیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (isset($period_analysis['city_usage']) && is_array($period_analysis['city_usage'])): ?>
                            <?php foreach ($period_analysis['city_usage'] as $item): ?>
                                <?php if (isset($item['city']) && isset($item['usage_section']) && isset($item['customer_count'])): ?>
                                <tr>
                                    <td><?= htmlspecialchars($item['city']) ?></td>
                                    <td><?= htmlspecialchars($item['usage_section']) ?></td>
                                    <td><?= number_format($item['customer_count']) ?></td>
                                    <td>
                                        <a href="<?= $back_url ?>&show_customers=1&category_type=usage_section&category_value=<?= urlencode($item['usage_section']) ?>#customer-list"
                                            class="btn btn-sm btn-outline-primary">
                                            نمایش لیست مشتریان
                                        </a>
                                    </td>
                                </tr>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="4" class="text-center text-muted">داده‌ای برای نمایش وجود ندارد</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- 7. نمودار کاربری - فرآورده -->
    <div class="analysis-header" onclick="toggleAnalysis('usage-product-<?= $period_name ?>')">
        <h5>
            <span>۷. نمودار کاربری (بخش مصرف) - فرآورده</span>
            <i class="bi bi-chevron-down toggle-icon" id="usage-product-<?= $period_name ?>-icon"></i>
        </h5>
    </div>
    <div class="analysis-content" id="usage-product-<?= $period_name ?>">
        <div class="chart-container">
            <canvas id="usageProductChart-<?= str_replace(' ', '-', $period_name) ?>"></canvas>
        </div>
        <div class="dataset-table">
            <div class="table-responsive">
                <table class="table table-bordered table-striped">
                    <thead class="table-dark">
                        <tr>
                            <th>بخش مصرف</th>
                            <th>فراورده</th>
                            <th>میزان مصرف</th>
                            <th>عملیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (isset($period_analysis['usage_product']) && is_array($period_analysis['usage_product'])): ?>
                            <?php foreach ($period_analysis['usage_product'] as $item): ?>
                                <?php if (isset($item['usage_section']) && isset($item['product']) && isset($item['customer_count'])): ?>
                                <tr>
                                    <td><?= htmlspecialchars($item['usage_section']) ?></td>
                                    <td><?= htmlspecialchars($item['product']) ?></td>
                                    <td><?= number_format($item['customer_count']) ?></td>
                                    <td>
                                        <a href="<?= $back_url ?>&show_customers=1&category_type=usage_section&category_value=<?= urlencode($item['usage_section']) ?>#customer-list"
                                            class="btn btn-sm btn-outline-primary">
                                            نمایش لیست مشتریان
                                        </a>
                                    </td>
                                </tr>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="4" class="text-center text-muted">داده‌ای برای نمایش وجود ندارد</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
                            </div>

        </div>
    <?php endforeach; ?>
<?php endif; ?>
<?php
// دریافت لیست مشتریان بر اساس فیلترها
$customer_list = [];
if ($show_customers && $selected_shamsi_year) {
    // ساخت شرط‌های WHERE
    $where_conditions = [];
    $params = [];
    
    // شرط زمانی (استفاده از اولین دوره زمانی)
    if (!empty($periods_data)) {
        $first_period = $periods_data[0];
        $where_conditions[] = "تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
        $params[] = $first_period['start'];
        $params[] = $first_period['end'];
    }
    
    // فیلتر شهرستان
    if ($selected_city) {
        $where_conditions[] = "شهرستان = ?";
        $params[] = $selected_city;
    }
    
    // فیلتر فراورده
    if ($selected_product) {
        $where_conditions[] = "فراوده = ?";
        $params[] = $selected_product;
    }
    
    // فیلتر بخش مصرف
    if ($selected_usage_section) {
        $where_conditions[] = "بخش_مصرف = ?";
        $params[] = $selected_usage_section;
    }
    
    // فیلتر نوع دسته‌بندی
    if ($category_type && $category_value) {
        $column_mapping = [
            'city' => 'شهرستان',
            'region' => 'استان',
            'usage_section' => 'بخش_مصرف',
            'product' => 'فراوده'
        ];
        
        if (isset($column_mapping[$category_type])) {
            $column_name = $column_mapping[$category_type];
            $where_conditions[] = "{$column_name} = ?";
            $params[] = $category_value;
        }
    }
    
    // ساخت کوئری نهایی
    $where_sql = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
    
    $query = "
        SELECT DISTINCT
            شناسه_یکتای_کاربرد,
            شماره_درخواست,
            نام_متقاضی,
            کدملی_شناسه_ملی,
            تلفن_همراه,
            استان,
            شهرستان,
            بخش_مصرف,
            فراوده
        FROM fuel_requests 
        {$where_sql}
        ORDER BY شهرستان, نام_متقاضی
    ";
    
    try {
        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        $customer_list = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching customer list: " . $e->getMessage());
        $customer_list = [];
    }
}
$back_url = "?";
if ($selected_shamsi_year) $back_url .= "shamsi_year=" . urlencode($selected_shamsi_year) . "&";
if ($time_period) $back_url .= "time_period=" . urlencode($time_period) . "&";
if ($selected_city) $back_url .= "city=" . urlencode($selected_city) . "&";
if ($selected_product) $back_url .= "product=" . urlencode($selected_product) . "&";
if ($selected_usage_section) $back_url .= "usage_section=" . urlencode($selected_usage_section) . "&";
if ($category_type) $back_url .= "category_type=" . urlencode($category_type) . "&";
if ($category_value) $back_url .= "category_value=" . urlencode($category_value);

// حذف & اضافی از انتهای URL
$back_url = rtrim($back_url, "&");
?>
                            <!-- بخش نمایش لیست مشتریان -->
                            <?php if ($show_customers && !empty($customer_list)): ?>
                                <div id="customer-list" class="mt-5">
                                    <div class="card">
                                        <div class="card-header bg-info text-white">
                                            <h5 class="mb-0">
                                                لیست مشتریان -
                                                <?php
                                                switch ($category_type) {
                                                    case 'city':
                                                        echo "شهرستان: ";
                                                        break;
                                                    case 'region':
                                                        echo "استان: ";
                                                        break;
                                                    case 'usage_section':
                                                        echo "بخش مصرف: ";
                                                        break;
                                                    case 'product':
                                                        echo "فراورده: ";
                                                        break;
                                                }
                                                echo $category_value;
                                                ?>
                                                <span class="badge bg-light text-dark"><?= count($customer_list) ?> مشتری</span>
                                            </h5>
                                             <a href="<?=$back_url?>"
                                                class="btn btn-secondary mt-3">بازگشت به تحلیل</a>
                                        </div>
                                        <div class="card-body">
                                            <div class="table-responsive">
                                                <table class="table table-bordered table-striped">
                                                    <thead class="table-dark">
                                                        <tr>
                                                              <th>شماره درخواست</th>
                                                       
                                                            <th>شناسه یکتا</th>
                                                            <th>نام متقاضی</th>
                                                            <th>کدملی/شناسه ملی</th>
                                                            <th>تلفن همراه</th>
                                                            <th>استان</th>
                                                            <th>شهرستان</th>
                                                            <th>بخش مصرف</th>
                                                            <th>فراورده</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($customer_list as $customer): ?>
                                                            <tr>
                                                                  <td><?= $customer['شماره_درخواست'] ?></td>
                                                          
                                                                <td><?= $customer['شناسه_یکتای_کاربرد'] ?></td>
                                                                <td><?= $customer['نام_متقاضی'] ?></td>
                                                                <td><?= $customer['کدملی_شناسه_ملی'] ?></td>
                                                                <td><?= $customer['تلفن_همراه'] ?></td>
                                                                <td><?= $customer['استان'] ?></td>
                                                                <td><?= $customer['شهرستان'] ?></td>
                                                                <td><?= $customer['بخش_مصرف'] ?></td>
                                                                <td><?= $customer['فراوده'] ?></td>
                                                            </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                            </div>
                                            <a href="<?=$back_url?>"
                                                class="btn btn-secondary mt-3">بازگشت به تحلیل</a>
                                        </div>
                                    </div>
                                </div>
                            <?php endif; ?>


                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- اضافه کردن آیکون‌های Bootstrap Icons -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">
<script>
    // توابع کنترل کشویی‌ها
    function toggleYear(yearId) {
        const section = document.getElementById(yearId);
        const icon = document.getElementById(yearId + '-icon');
        if (section && icon) {
            section.classList.toggle('show');
            icon.classList.toggle('rotated');
        }
    }

    function toggleAnalysis(analysisId) {
        const content = document.getElementById(analysisId);
        const icon = document.getElementById(analysisId + '-icon');
        if (content && icon) {
            content.classList.toggle('show');
            icon.classList.toggle('rotated');
        }
    }

    function expandAll() {
        document.querySelectorAll('.analysis-content').forEach(content => {
            content.classList.add('show');
        });
        document.querySelectorAll('.toggle-icon').forEach(icon => {
            icon.classList.add('rotated');
        });
    }

    function collapseAll() {
        document.querySelectorAll('.analysis-content').forEach(content => {
            content.classList.remove('show');
        });
        document.querySelectorAll('.toggle-icon').forEach(icon => {
            icon.classList.remove('rotated');
        });
    }

    // تولید رنگ‌های تصادفی
    function generateColors(count) {
        const colors = [];
        for (let i = 0; i < count; i++) {
            const hue = (i * 137.5) % 360;
            colors.push(`hsl(${hue}, 70%, 60%)`);
        }
        return colors;
    }

    // تابع ایجاد نمودار برای یک دوره خاص
    function createChartsForPeriod(periodName, periodData) {
        const colors = generateColors(50);
        const periodId = periodName.replace(/ /g, '-');
        
        // 2. نمودار ناحیه - شهرستان
        const regionCityCtx = document.getElementById(`regionCityChart-${periodId}`);
        if (regionCityCtx && periodData.region_city) {
            new Chart(regionCityCtx, {
                type: 'pie',
                data: {
                    labels: periodData.region_city.map(item => `${item.region} - ${item.city}`),
                    datasets: [{
                        data: periodData.region_city.map(item => item.customer_count),
                        backgroundColor: colors,
                        borderWidth: 2,
                        borderColor: '#fff'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: `توزیع مصرف بر اساس استان و شهرستان - ${periodName}`
                        },
                        legend: {
                            position: 'left',
                            rtl: true
                        }
                    }
                }
            });
        }

        // 3.نمودار میزان مصرف- ناحیه (شهرستان)
        const cityCustomersCtx = document.getElementById(`cityCustomersChart-${periodId}`);
        if (cityCustomersCtx && periodData.city_customers) {
            new Chart(cityCustomersCtx, {
                type: 'pie',
                data: {
                    labels: periodData.city_customers.map(item => item.city),
                    datasets: [{
                        data: periodData.city_customers.map(item => item.customer_count),
                        backgroundColor: colors,
                        borderWidth: 2,
                        borderColor: '#fff'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: `توزیع مصرف بر اساس شهرستان - ${periodName}`
                        },
                        legend: {
                            position: 'left',
                            rtl: true
                        }
                    }
                }
            });
        }

        // 5. نمودار ناحیه - کاربری
        const regionUsageCtx = document.getElementById(`regionUsageChart-${periodId}`);
        if (regionUsageCtx && periodData.region_usage) {
            new Chart(regionUsageCtx, {
                type: 'pie',
                data: {
                    labels: periodData.region_usage.map(item => `${item.region} - ${item.usage_section}`),
                    datasets: [{
                        data: periodData.region_usage.map(item => item.customer_count),
                        backgroundColor: colors,
                        borderWidth: 2,
                        borderColor: '#fff'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: `توزیع بخش‌های مصرف در استان‌ها - ${periodName}`
                        },
                        legend: {
                            position: 'left',
                            rtl: true
                        }
                    }
                }
            });
        }

        // 4. نمودار شهرستان‌ها - فراورده (مقدار تایید شده)
        const cityProductCtx = document.getElementById(`cityProductChart-${periodId}`);
        if (cityProductCtx && periodData.city_product) {
            const cityProductData = periodData.city_product;

            // گروه‌بندی داده‌ها بر اساس شهرستان
            const cityGroups = {};
            cityProductData.forEach(item => {
                if (!cityGroups[item.city]) {
                    cityGroups[item.city] = [];
                }
                const quantity = item.total_quantity || 0;
                cityGroups[item.city].push({
                    product: item.product,
                    quantity: quantity,
                    customer_count: item.customer_count || 0
                });
            });

            // ایجاد select برای انتخاب شهرستان
            const citySelector = document.createElement('select');
            citySelector.className = 'form-select mb-3';
            citySelector.innerHTML = `
                <option value="">انتخاب شهرستان برای مشاهده نمودار فراورده‌ها</option>
                ${Object.keys(cityGroups).map(city => `<option value="${city}">${city}</option>`).join('')}
            `;

            // قرار دادن select قبل از نمودار
            cityProductCtx.parentNode.insertBefore(citySelector, cityProductCtx);

            // تابع ایجاد نمودار برای شهرستان انتخاب شده
            function createCityProductChart(selectedCity = '') {
                // حذف نمودار قبلی اگر وجود دارد
                if (window[`cityProductChartInstance-${periodId}`]) {
                    window[`cityProductChartInstance-${periodId}`].destroy();
                }

                let chartData, chartTitle;

                if (selectedCity && cityGroups[selectedCity]) {
                    // نمودار برای شهرستان خاص
                    const cityData = cityGroups[selectedCity];
                    chartData = {
                        labels: cityData.map(item => item.product),
                        datasets: [{
                            data: cityData.map(item => item.quantity),
                            backgroundColor: generateColors(cityData.length),
                            borderWidth: 2,
                            borderColor: '#fff'
                        }]
                    };
                    chartTitle = `توزیع فراورده‌های تایید شده در شهرستان ${selectedCity} - ${periodName}`;
                } else {
                    // نمودار کلی برای همه شهرستان‌ها
                    const productTotals = {};
                    cityProductData.forEach(item => {
                        const quantity = item.total_quantity || 0;
                        if (!productTotals[item.product]) {
                            productTotals[item.product] = 0;
                        }
                        productTotals[item.product] += quantity;
                    });

                    const sortedProducts = Object.entries(productTotals)
                        .sort((a, b) => b[1] - a[1])
                        .slice(0, 10);

                    chartData = {
                        labels: sortedProducts.map(item => item[0]),
                        datasets: [{
                            data: sortedProducts.map(item => item[1]),
                            backgroundColor: generateColors(sortedProducts.length),
                            borderWidth: 2,
                            borderColor: '#fff'
                        }]
                    };
                    chartTitle = `توزیع فراورده‌های تایید شده در همه شهرستان‌ها - ${periodName}`;
                }

                // ایجاد نمودار جدید
                window[`cityProductChartInstance-${periodId}`] = new Chart(cityProductCtx, {
                    type: 'pie',
                    data: chartData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            title: {
                                display: true,
                                text: chartTitle
                            },
                            legend: {
                                position: 'left',
                                rtl: true
                            },
                            tooltip: {
                                callbacks: {
                                    label: function (context) {
                                        const label = context.label || '';
                                        const value = context.raw || 0;
                                        const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                        const percentage = total > 0 ? Math.round((value / total) * 100) : 0;
                                        return `${label}: ${value.toLocaleString()} لیتر (${percentage}%)`;
                                    }
                                }
                            }
                        }
                    }
                });
            }

            // ایجاد نمودار اولیه
            createCityProductChart();

            // اضافه کردن event listener برای تغییر شهرستان
            citySelector.addEventListener('change', function () {
                createCityProductChart(this.value);
            });
        }

        // 6. نمودار شهرستان - کاربری
        const cityUsageCtx = document.getElementById(`cityUsageChart-${periodId}`);
        if (cityUsageCtx && periodData.city_usage) {
            new Chart(cityUsageCtx, {
                type: 'doughnut',
                data: {
                    labels: periodData.city_usage.map(item => `${item.city} - ${item.usage_section}`).slice(0, 12),
                    datasets: [{
                        data: periodData.city_usage.map(item => item.customer_count).slice(0, 12),
                        backgroundColor: colors,
                        borderWidth: 2,
                        borderColor: '#fff'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: `توزیع بخش‌های مصرف در شهرستان‌ها - ${periodName}`
                        },
                        legend: {
                            position: 'left',
                            rtl: true
                        }
                    }
                }
            });
        }

        // 7. نمودار کاربری - فرآورده
        const usageProductCtx = document.getElementById(`usageProductChart-${periodId}`);
        if (usageProductCtx && periodData.usage_product) {
            new Chart(usageProductCtx, {
                type: 'doughnut',
                data: {
                    labels: periodData.usage_product.map(item => `${item.usage_section} - ${item.product}`).slice(0, 10),
                    datasets: [{
                        data: periodData.usage_product.map(item => item.customer_count).slice(0, 10),
                        backgroundColor: colors,
                        borderWidth: 2,
                        borderColor: '#fff'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: `توزیع فراورده‌ها در بخش‌های مصرف مختلف - ${periodName}`
                        },
                        legend: {
                            position: 'left',
                            rtl: true
                        }
                    }
                }
            });
        }
    }

    // ایجاد نمودارها برای تمام دوره‌ها
    document.addEventListener('DOMContentLoaded', function () {
        <?php if (!empty($analysis_data)): ?>
            <?php foreach($analysis_data as $period_name => $period_data): ?>
                createChartsForPeriod('<?= $period_name ?>', <?= json_encode($period_data) ?>);
            <?php endforeach; ?>
        <?php endif; ?>

        // حالت پیش‌فرض: بستن همه تحلیل‌ها
        collapseAll();
    });
</script>
</body>

</html>