<?php
// product_distribution.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;

// دریافت پارامترهای فیلتر
$city_filter = $_GET['city_filter'] ?? '';
$product_filter = $_GET['product_filter'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// تبدیل تاریخ‌های شمسی به میلادی برای جستجو
if (!empty($date_from)) {
    try {
        if (preg_match('/^(\d{4})\/(\d{1,2})\/(\d{1,2})$/', $date_from, $matches)) {
            $jalali = Jalalian::fromFormat('Y/m/d', $date_from);
            $date_from = $jalali->toCarbon()->format('Y-m-d');
        }
    } catch (Exception $e) {
        $date_from = '';
    }
}

if (!empty($date_to)) {
    try {
        if (preg_match('/^(\d{4})\/(\d{1,2})\/(\d{1,2})$/', $date_to, $matches)) {
            $jalali = Jalalian::fromFormat('Y/m/d', $date_to);
            $date_to = $jalali->toCarbon()->format('Y-m-d');
        }
    } catch (Exception $e) {
        $date_to = '';
    }
}

// ساخت شرط‌های WHERE
$where_conditions = [];
$params = [];

if (!empty($city_filter)) {
    $where_conditions[] = "شهرستان = ?";
    $params[] = $city_filter;
}

if (!empty($product_filter)) {
    $where_conditions[] = "فراوده = ?";
    $params[] = $product_filter;
}

if (!empty($date_from)) {
    $where_conditions[] = "تاریخ_ثبت_درخواست >= ?";
    $params[] = $date_from;
}

if (!empty($date_to)) {
    $where_conditions[] = "تاریخ_ثبت_درخواست <= ?";
    $params[] = $date_to;
}

$where_sql = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// 1. توزیع بر اساس شهرستان
$distribution_city = $pdo->prepare("
    SELECT 
        شهرستان,
        SUM(مقدار_تایید_شده_توسط_پخش) as total_amount,
        COUNT(DISTINCT شناسه_یکتای_کاربرد) as consumer_count,
        ROUND(SUM(مقدار_تایید_شده_توسط_پخش) / COUNT(DISTINCT شناسه_یکتای_کاربرد), 2) as avg_per_consumer
    FROM fuel_requests 
    $where_sql
    GROUP BY شهرستان
    ORDER BY total_amount DESC
");
$distribution_city->execute($params);
$city_distribution = $distribution_city->fetchAll();

// 2. توزیع بر اساس فراورده
$distribution_product = $pdo->prepare("
    SELECT 
        فراوده as product,
        SUM(مقدار_تایید_شده_توسط_پخش) as total_amount,
        COUNT(DISTINCT شناسه_یکتای_کاربرد) as consumer_count,
        ROUND(SUM(مقدار_تایید_شده_توسط_پخش) / COUNT(DISTINCT شناسه_یکتای_کاربرد), 2) as avg_per_consumer
    FROM fuel_requests 
    $where_sql
    GROUP BY فراوده
    ORDER BY total_amount DESC
");
$distribution_product->execute($params);
$product_distribution = $distribution_product->fetchAll();

// 3. توزیع ترکیبی
$distribution_combined = $pdo->prepare("
    SELECT 
        شهرستان,
        فراوده as product,
        SUM(مقدار_تایید_شده_توسط_پخش) as total_amount,
        COUNT(DISTINCT شناسه_یکتای_کاربرد) as consumer_count,
        ROUND(SUM(مقدار_تایید_شده_توسط_پخش) / COUNT(DISTINCT شناسه_یکتای_کاربرد), 2) as avg_per_consumer
    FROM fuel_requests 
    $where_sql
    GROUP BY شهرستان, فراوده
    ORDER BY شهرستان, total_amount DESC
");
$distribution_combined->execute($params);
$combined_distribution = $distribution_combined->fetchAll();

// دریافت لیست‌ها برای فیلتر
$cities = $pdo->query("SELECT DISTINCT شهرستان FROM fuel_requests WHERE شهرستان IS NOT NULL ORDER BY شهرستان")->fetchAll(PDO::FETCH_COLUMN);
$products = $pdo->query("SELECT DISTINCT فراوده FROM fuel_requests WHERE فراوده IS NOT NULL ORDER BY فراوده")->fetchAll(PDO::FETCH_COLUMN);
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>توزیع فراورده</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://unpkg.com/persian-datepicker@1.2.0/dist/css/persian-datepicker.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://unpkg.com/persian-datepicker@1.2.0/dist/js/persian-datepicker.min.js"></script>
    <style>
        body { font-family: 'Tahoma', 'Arial', sans-serif; background-color: #f8f9fa; }
        .stat-card { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border-radius: 10px; }
        .stat-card h3 { font-size: 2rem; margin-bottom: 0; }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-12">
                <div class="card shadow">
                    <div class="card-header bg-success text-white">
                        <h4 class="mb-0">توزیع فراورده</h4>
                    </div>
                    <div class="card-body">
                        <!-- فیلترها -->
                        <form method="GET" class="row g-3 mb-4">
                            <div class="col-md-3">
                                <label class="form-label">شهرستان</label>
                                <select class="form-select" name="city_filter">
                                    <option value="">همه شهرستان‌ها</option>
                                    <?php foreach($cities as $city): ?>
                                        <option value="<?= $city ?>" <?= $city == $city_filter ? 'selected' : '' ?>><?= $city ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">فراورده</label>
                                <select class="form-select" name="product_filter">
                                    <option value="">همه فراورده‌ها</option>
                                    <?php foreach($products as $product): ?>
                                        <option value="<?= $product ?>" <?= $product == $product_filter ? 'selected' : '' ?>><?= $product ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">از تاریخ (شمسی)</label>
                                <input type="text" class="form-control" name="date_from" id="date_from" value="<?= htmlspecialchars($_GET['date_from'] ?? '') ?>" placeholder="1403/01/01">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">تا تاریخ (شمسی)</label>
                                <input type="text" class="form-control" name="date_to" id="date_to" value="<?= htmlspecialchars($_GET['date_to'] ?? '') ?>" placeholder="1403/12/29">
                            </div>
                            <div class="col-12">
                                <button type="submit" class="btn btn-success">اعمال فیلترها</button>
                                <a href="product_distribution.php" class="btn btn-secondary">پاک کردن فیلترها</a>
                            </div>
                        </form>

                        <!-- آمار کلی -->
                        <div class="row mb-4">
                            <div class="col-md-3">
                                <div class="stat-card p-3 text-center">
                                    <h3><?= number_format(array_sum(array_column($city_distribution, 'total_amount')), 2) ?></h3>
                                    <p class="mb-0">کل مقدار توزیع شده</p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stat-card p-3 text-center">
                                    <h3><?= number_format(array_sum(array_column($city_distribution, 'consumer_count'))) ?></h3>
                                    <p class="mb-0">تعداد مصرف‌کنندگان</p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stat-card p-3 text-center">
                                    <h3><?= count($city_distribution) ?></h3>
                                    <p class="mb-0">تعداد شهرستان‌ها</p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stat-card p-3 text-center">
                                    <h3><?= count($product_distribution) ?></h3>
                                    <p class="mb-0">تعداد فراورده‌ها</p>
                                </div>
                            </div>
                        </div>

                        <!-- نمودارها -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-header">توزیع بر اساس شهرستان</div>
                                    <div class="card-body">
                                        <canvas id="cityDistributionChart" height="300"></canvas>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-header">توزیع بر اساس فراورده</div>
                                    <div class="card-body">
                                        <canvas id="productDistributionChart" height="300"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- جداول -->
                        <div class="row mt-4">
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-header">رتبه‌بندی شهرستان‌ها</div>
                                    <div class="card-body">
                                        <div class="table-responsive">
                                            <table class="table table-striped">
                                                <thead>
                                                    <tr>
                                                        <th>رتبه</th>
                                                        <th>شهرستان</th>
                                                        <th>مقدار توزیع</th>
                                                        <th>تعداد مصرف‌کننده</th>
                                                        <th>میانگین به ازای هر مصرف‌کننده</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php foreach($city_distribution as $index => $row): ?>
                                                    <tr>
                                                        <td><?= $index + 1 ?></td>
                                                        <td><?= $row['شهرستان'] ?></td>
                                                        <td><?= number_format($row['total_amount'], 2) ?></td>
                                                        <td><?= $row['consumer_count'] ?></td>
                                                        <td><?= number_format($row['avg_per_consumer'], 2) ?></td>
                                                    </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-header">رتبه‌بندی فراورده‌ها</div>
                                    <div class="card-body">
                                        <div class="table-responsive">
                                            <table class="table table-striped">
                                                <thead>
                                                    <tr>
                                                        <th>رتبه</th>
                                                        <th>فراورده</th>
                                                        <th>مقدار توزیع</th>
                                                        <th>تعداد مصرف‌کننده</th>
                                                        <th>میانگین به ازای هر مصرف‌کننده</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php foreach($product_distribution as $index => $row): ?>
                                                    <tr>
                                                        <td><?= $index + 1 ?></td>
                                                        <td><?= $row['product'] ?></td>
                                                        <td><?= number_format($row['total_amount'], 2) ?></td>
                                                        <td><?= $row['consumer_count'] ?></td>
                                                        <td><?= number_format($row['avg_per_consumer'], 2) ?></td>
                                                    </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // نمودار توزیع شهرستان‌ها
        const cityDistCtx = document.getElementById('cityDistributionChart').getContext('2d');
        new Chart(cityDistCtx, {
            type: 'pie',
            data: {
                labels: <?= json_encode(array_column($city_distribution, 'شهرستان')) ?>,
                datasets: [{
                    data: <?= json_encode(array_column($city_distribution, 'total_amount')) ?>,
                    backgroundColor: [
                        '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF',
                        '#FF9F40', '#FF6384', '#C9CBCF', '#4BC0C0', '#FF6384'
                    ]
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'left',
                    },
                    title: {
                        display: true,
                        text: 'سهم شهرستان‌ها از کل توزیع'
                    }
                }
            }
        });

        // نمودار توزیع فراورده‌ها
        const productDistCtx = document.getElementById('productDistributionChart').getContext('2d');
        new Chart(productDistCtx, {
            type: 'doughnut',
            data: {
                labels: <?= json_encode(array_column($product_distribution, 'product')) ?>,
                datasets: [{
                    data: <?= json_encode(array_column($product_distribution, 'total_amount')) ?>,
                    backgroundColor: [
                        '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF',
                        '#FF9F40', '#FF6384', '#C9CBCF', '#4BC0C0', '#FF6384'
                    ]
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'left',
                    },
                    title: {
                        display: true,
                        text: 'سهم فراورده‌ها از کل توزیع'
                    }
                }
            }
        });

        // فعال‌سازی تقویم شمسی
        $(document).ready(function() {
            $("#date_from").persianDatepicker({
                format: 'YYYY/MM/DD',
                observer: true,
                altField: '.observer-example-alt',
                altFormat: 'YYYY/MM/DD'
            });
            $("#date_to").persianDatepicker({
                format: 'YYYY/MM/DD',
                observer: true,
                altField: '.observer-example-alt',
                altFormat: 'YYYY/MM/DD'
            });
        });
    </script>
</body>
</html>