<?php

namespace Modules\Payment\Http\Controllers\Front;

use App\Jobs\SendSms;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Address\Entities\Address;
use Modules\Cart\Helpers\Cart;
use Modules\Payment\Entities\Payment;
use Modules\Payment\Notifications\NewOrderAdminNotification;
use Modules\Payment\Notifications\PaymentSuccessNotification;
use Modules\User\Entities\User;
use Shetabit\Multipay\Exceptions\InvalidPaymentException;
use Shetabit\Multipay\Invoice;
use Shetabit\Payment\Facade\Payment as ShetabitPayment;

class PaymentController extends Controller
{
    public function payment(Request $request)
    {
        $cookieCart = Cart::instance('cart-ic-world');
        $cartItems = $cookieCart->all();
        foreach ($cartItems as $cartItem) {
            if (!is_null($cartItem['product']->inventory)) {
                if ($cartItem['product']->inventory === 0 || $cartItem['quantity'] > $cartItem['product']->inventory) {
                    $productWithZeroInventory[] = $cartItem['product'];
                    $cookieCart->delete($cartItem['id']);
                }
            }
        }
        if (isset($productWithZeroInventory)) {
            if (count($productWithZeroInventory) > 0) {
                return view('cart::frontend.cart', compact(['productWithZeroInventory', 'cookieCart']));
            }
        }
        if ($cartItems->count()) {
            $price = $cartItems->sum(function ($cart) {
                if ($cart['discount_percent'] == 0) {
                    if (!is_null($cart['product']->sale_price)) {
                        return $cart['product']->sale_price * $cart['quantity'];
                    } else {
                        return $cart['product']->price * $cart['quantity'];
                    }
                } else {
                    if (!is_null($cart['product']->sale_price)) {
                        return $cart['product']->sale_price * $cart['quantity'];
                    } else {
                        return ($cart['product']->price - ($cart['product']->price * $cart['discount_percent'])) * $cart['quantity'];
                    }
                }
            });

            $orderItems = $cartItems->mapWithKeys(function ($cart) {

                return [$cart['product']->id => ['quantity' => $cart['quantity'], 'price' => $cart['price']]];
            });

            if ($request->input('address-id') && $request->input('address')){
                $request->request->remove('address-id');
            }

            if (!$request->input('address-id')) {
                $validData = $request->validate([
                    'name' => 'required',
                    'last_name' => 'required',
                    'company' => 'nullable',
                    'city' => 'required',
                    'address' => 'required',
                    'address-id' => 'nullable',
                    'postal_code' => 'required' , 'regex:\b(?!(\d)\1{3})[13-9]{4}[1346-9][013-9]{5}\b',
                    'notes' => 'nullable',
                    'shipping' => 'required'
                ]);
                $address = auth()->user()->addresses()->create([
                    'city_id' => $validData['city'],
                    'address' => $validData['address'],
                    'postal_code' => $validData['postal_code']
                ]);
                auth()->user()->update([
                    'name' => $validData['name'],
                    'last_name' => $validData['last_name'],
                ]);
                $order = auth()->user()->orders()->create([
                    'status' => 'unpaid',
                    'price' => $price,
                    'shipping_id' => $validData['shipping'],
                    'address_id' => $address->id,
                    'notes' => $validData['notes'],
                    'order_code' => rand(1000 , 9999),
                ]);
            }else {
                $validData = $request->validate([
                    'address-id' => 'nullable',
                    'note' => 'nullable',
                    'shipping' => 'required',
                    'notes' => 'nullable'
                ]);
                $address = Address::whereId($validData['address-id'])->first();
                $order = auth()->user()->orders()->create([
                    'status' => 'unpaid',
                    'price' => $price,
                    'shipping_id' => $validData['shipping'],
                    'address_id' => $address->id,
                    'order_code' => rand(1000 , 9999),
                    'notes' => $validData['notes']
                ]);
            }

            $order->products()->attach($orderItems);
            $cookieCart->flush();
            
            $admins = User::where('is_admin' , 1)->get();
            
            foreach ($admins as $admin) {
                $order->notify(new NewOrderAdminNotification(
                    $admin->phone ,
                    $order->order_code ,
                    $order->price ,
                    $order->user->name ,
                    $order->address->city->name ,
                    $order->shipping->label
                ));
            }
            return view('payment::cardToCard' , compact('order'));
        }

        alert()->error('خطا', 'متاسفانه مشکلی در ثبت سفارش شما پیش آمده، لطفا دوباره تلاش کنید');
        return back();
    }

    public function callback(Request $request)
    {
        try {
            $payment = Payment::where('resnumber', $request->Authority)->firstOrFail();
            // $payment->order->price
            $receipt = ShetabitPayment::amount($payment->order->price)->transactionId($request->Authority)->verify();

            $payment->update([
                'status' => 1
            ]);

            foreach ($payment->order->products as $product) {
                if (!is_null($product->inventory)) {
                    $product->update([
                        'inventory' => $product->inventory - $product->pivot->quantity
                    ]);
                }
            }
            $payment->order()->update([
                'status' => 'paid'
            ]);

            alert()->success('پرداخت شما موفق بود');
            return redirect(route('profile.orders'));

        } catch (InvalidPaymentException $exception) {
            /**
             * when payment is not verified, it will throw an exception.
             * We can catch the exception to handle invalid payments.
             * getMessage method, returns a suitable message that can be used in user interface.
             **/
            alert()->error($exception->getMessage());
            return redirect(route('profile.orders'));
        }
    }
}
