<?php
/**
 * Siaracorporatebusiness functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Siaracorporatebusiness
 */

$theme_version = wp_get_theme()->get( 'Version' );
if ( ! defined( '_S_VERSION' ) ) {
	define( '_S_VERSION', $theme_version );
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function siaracorporatebusiness_setup() {
	/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on Siaracorporatebusiness, use a find and replace
		* to change 'siara-corporate-business' to the name of your theme in all the template files.
		*/
	load_theme_textdomain( 'siara-corporate-business', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
	add_theme_support( 'title-tag' );

	/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
	add_theme_support( 'post-thumbnails' );

	/**
		 * Add post-formats support.
		 */
	add_theme_support(
		'post-formats',
		array(
			'gallery',
			'video',
			'audio',
		)
	);

	register_nav_menus(
		array(
			'top-menu'     => esc_html__( 'Top Menu', 'siara-corporate-business' ),
			'primary-menu' => esc_html__( 'Primary Menu', 'siara-corporate-business' ),
			'social-menu'  => esc_html__( 'Social Menu', 'siara-corporate-business' ),
			'footer-menu'  => esc_html__( 'Footer Menu', 'siara-corporate-business' ),
		)
	);

	/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background',
		apply_filters(
			'siaracorporatebusiness_custom_background_args',
			array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support(
		'custom-logo',
		array(
			'height'               => 100,
			'width'                => 350,
			'flex-width'           => true,
			'flex-height'          => true,
			'unlink-homepage-logo' => false,
		)
	);

	add_image_size( 'siaracorporatebusiness-cover-image', 1280, 1000, true );
	add_image_size( 'siaracorporatebusiness-large-img', 800, 450, true );
	add_image_size( 'siaracorporatebusiness-medium-img', 750, 575, true );
	add_image_size( 'siaracorporatebusiness-square-img', 500, 500, true );

	// Theme supports wide images, galleries and videos.
	add_theme_support( 'align-wide' );
	add_theme_support( 'responsive-embeds' );
	add_theme_support( 'wp-block-styles' );
}
add_action( 'after_setup_theme', 'siaracorporatebusiness_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function siaracorporatebusiness_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'siaracorporatebusiness_content_width', 640 );
}
add_action( 'after_setup_theme', 'siaracorporatebusiness_content_width', 0 );

function siaracorporatebusiness_custom_styles() {
    wp_enqueue_style(
        'siaracorporatebusiness-custom-style',
        get_template_directory_uri() . '/assets/custom/css/frontpage-design1/style.css', 
        array(), 
        _S_VERSION, 
        'all' 
    );
}
add_action('wp_enqueue_scripts', 'siaracorporatebusiness_custom_styles');

/**
 * Get font url
 */
if ( ! function_exists( 'siaracorporatebusiness_font_url' ) ) :
	/**
	 * Return google font URL.
	 *
	 * @since 2.0.0
	 * @return string Font URL.
	 */
	function siaracorporatebusiness_font_url() {

		$font_url = '';
		$fonts    = $fonts_name = array();

		$primary_font      = get_theme_mod( 'primary_font', '"Lora", "regular:500:600:700:italic:500italic:600italic:700italic", serif' );
		$primary_menu_font = get_theme_mod( 'primary_menu_font', '"Work Sans", "100:200:300:regular:500:600:700:800:900:100italic:200italic:300italic:italic:500italic:600italic:700italic:800italic:900italic", sans-serif' );
		$sub_menu_font     = get_theme_mod( 'sub_menu_font', '"Work Sans", "100:200:300:regular:500:600:700:800:900:100italic:200italic:300italic:italic:500italic:600italic:700italic:800italic:900italic", sans-serif' );
		$secondary_font    = get_theme_mod( 'secondary_font', '"Work Sans", "100:200:300:regular:500:600:700:800:900:100italic:200italic:300italic:italic:500italic:600italic:700italic:800italic:900italic", sans-serif' );

		$fonts[] = $primary_font;
		$fonts[] = $primary_menu_font;
		$fonts[] = $sub_menu_font;
		$fonts[] = $secondary_font;

		foreach ( $fonts as $font ) {
			$font_family         = str_replace( '"', '', $font );
			$font_family_explode = explode( ', ', $font_family );
			$fonts_name[]        = isset( $font_family_explode[0] ) ? $font_family_explode[0] : '';
		}

		// Check for same fonts
		$fonts_name = array_unique( $fonts_name );

		$font_final_string = implode( ':300,300italic,400,400italic,500,500italic,600,600italic,700,700italic,800,800italic|', $fonts_name );

		if ( 'off' !== _x( 'on', 'Google font: on or off', 'siara-corporate-business' ) ) {
			$font_url = add_query_arg(
				array(
					'family'  => urlencode( $font_final_string . ':300,300italic,400,400italic,500,500italic,600,600italic,700,700italic,800,800italic&subset=latin,cyrillic,cyrillic-ext,greek,greek-ext,latin-ext' ),
					'display' => 'swap',
				),
				'https://fonts.googleapis.com/css'
			);
		}

		return $font_url;
	}
endif;

/**
 * Enqueue scripts and styles.
 */
function siaracorporatebusiness_scripts() {

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	// Font Loader.
	require_once get_theme_file_path( 'lib/font-loader/wptt-webfont-loader.php' );

	// Styles.
	wp_enqueue_style( 'swiper', get_template_directory_uri() . '/assets/lib/swiper/swiper-bundle' . $min . '.css' );
	if ( is_child_theme() ) {
		wp_enqueue_style( 'siaracorporatebusiness-parent-style', trailingslashit( get_template_directory_uri() ) . 'style.css' );
	}
	wp_enqueue_style( 'siaracorporatebusiness-style', get_stylesheet_uri(), array(), _S_VERSION );
	wp_style_add_data( 'siaracorporatebusiness-style', 'rtl', 'replace' );
	wp_add_inline_style( 'siaracorporatebusiness-style', siaracorporatebusiness_get_inline_css() );

	// Widget css.
	if ( 'global' == get_theme_mod( 'widgets_css_loading', 'conditional' ) ) {
		$block_file = is_rtl() ? 'blocks-rtl' : 'blocks';
		wp_enqueue_style( 'siaracorporatebusiness-blocks-css', get_template_directory_uri() . '/assets/custom/css/' . $block_file . '.css' );
	}

	// Fonts.
	$font_url = siaracorporatebusiness_font_url();
	if ( ! empty( $font_url ) ) {
		wp_enqueue_style( 'siaracorporatebusiness-google-fonts', wptt_get_webfont_url( $font_url ), array(), null );
	}

	// Scripts
	$dependencies = array( 'swiper' );

	wp_enqueue_script( 'swiper', get_template_directory_uri() . '/assets/lib/swiper/swiper-bundle' . $min . '.js', array(), _S_VERSION, true );
	wp_enqueue_script( 'siaracorporatebusiness-script', get_template_directory_uri() . '/assets/custom/js/script' . $min . '.js', $dependencies, _S_VERSION, true );
	
	wp_enqueue_script( 'siaracorporatebusiness-swiper-script', get_template_directory_uri() . '/assets/custom/js/custom-swiper-script.js', array(), true );

	// Localized variables.
	global $wp_query;
	wp_localize_script(
		'siaracorporatebusiness-script',
		'SiaracorporatebusinessVars',
		array(
			'load_post_nonce' => wp_create_nonce( 'siaracorporatebusiness-load-posts-nonce' ),
			'ajaxurl'         => admin_url( 'admin-ajax.php' ),
			'query_vars'      => wp_json_encode( $wp_query->query_vars ),
		)
	);

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'siaracorporatebusiness_scripts' );

// Admin scripts and styles.
function siaracorporatebusiness_admin_scripts() {

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	wp_enqueue_style( 'siaracorporatebusiness-admin-style', get_template_directory_uri() . '/assets/custom/css/admin' . $min . '.css', false, _S_VERSION );
}
add_action( 'admin_enqueue_scripts', 'siaracorporatebusiness_admin_scripts' );

/**
 * Load all required files.
 */
require get_template_directory() . '/inc/init.php';


function siaracorporatebusiness_add_design_template_menu() {
    add_theme_page(
        __('Starter Template', 'siara-corporate-business'), // page title
        __('Starter Template', 'siara-corporate-business'), // menu title
        'edit_theme_options',                      // capability
        'siaracorporatebusiness-select-template',              // slug
        'siaracorporatebusiness_render_template_admin_page'    // callback function
    );
}
add_action('admin_menu', 'siaracorporatebusiness_add_design_template_menu');
 
function siaracorporatebusiness_render_template_admin_page() {
    // Handle form submission
    if (isset($_POST['siaracorporatebusiness_selected_template'])) {
        $selected_template = sanitize_text_field($_POST['siaracorporatebusiness_selected_template']);
        set_theme_mod('siaracorporatebusiness_homepage_template', $selected_template);
        echo '<div class="updated"><p><strong>Design template updated!</strong></p></div>';
    }
 
    $current_template = get_theme_mod('siaracorporatebusiness_homepage_template', 'front-page.php');
 
    $free_templates = array(
        'front-page.php' => 'Design 1',
    );
 
    ?>
 
    <div class="wrap">
        <h1><?php esc_html_e( 'Select Homepage Design', 'siara-corporate-business' ); ?></h1>
        <form method="post">
            <style>
                .template-option {
                    display: inline-block;
                    width: 20%;
                    margin: 20px;
                    border: 2px solid #ddd;
                    padding: 10px;
                    text-align: center;
                    background: #fff;
                    position: relative;
                }
                .template-option img {
                    max-width: 100%;
                    height: auto;
                    display: block;
                    margin-bottom: 10px;
                }
                .template-option.active {
                    border-color: #007cba;
                    box-shadow: 0 0 10px rgba(0, 124, 186, 0.4);
                }
                .template-option input[type="submit"] {
                    margin-top: 10px;
                }
                .template-option.locked::after {
                    content: "Pro";
                    position: absolute;
                    top: 10px;
                    left: 10px;
                    background: #d54e21;
                    color: #fff;
                    padding: 3px 8px;
                    font-size: 12px;
                    border-radius: 3px;
                }
            </style>
 
            <h2><?php esc_html_e( 'Free Templates', 'siara-corporate-business' ); ?></h2>
            <?php foreach ($free_templates as $file => $label): ?>
                <div class="template-option <?php echo $current_template === $file ? 'active' : ''; ?>">
					<img src="<?php echo esc_url( get_template_directory_uri() . '/assets/images/starter-template-1.png' ); ?>" alt="<?php echo esc_attr($label); ?>" />
                    <p><?php echo esc_html($label); ?></p>
                    <button type="submit" name="siaracorporatebusiness_selected_template" value="<?php echo esc_attr($file); ?>" class="button button-primary">
                        <?php echo $current_template === $file ? 'Activated' : 'Select & Publish'; ?>
                    </button>
                </div>
            <?php endforeach; ?>
 
        </form>
    </div>
    <?php
}