<?php
/**
 * Custom comment walker for this theme
 *
 * @package WordPress
 * @subpackage Radiance
 * @since 1.0.0
 */

/**
 * This class outputs custom comment walker for HTML5 friendly WordPress comment and threaded replies.
 *
 * @since 1.0.0
 */
class Radiance_Walker_Comment extends Walker_Comment {

	/**
	 * Outputs a comment in the HTML5 format.
	 *
	 * @see wp_list_comments()
	 *
	 * @param WP_Comment $comment Comment to display.
	 * @param int        $depth   Depth of the current comment.
	 * @param array      $args    An array of arguments.
	 */
	protected function html5_comment( $comment, $depth, $args ) {
		$tag = ( 'div' === $args['style'] ) ? 'div' : 'li';

		?>
		<<?php echo $tag; ?> id="comment-<?php comment_ID(); ?>" <?php comment_class( $this->has_children ? 'parent' : '', $comment ); ?>>
			<article id="div-comment-<?php comment_ID(); ?>" class="comment-body table-bordered mt-2 rounded">
				<footer class="comment-meta rounded p-2">
					<div class="comment-author vcard">
						<?php
						$comment_author_url = get_comment_author_url( $comment );
						$comment_author     = get_comment_author( $comment );
						$avatar             = get_avatar( $comment, $args['avatar_size'], '', '', ['class'=>'rounded'] );
						if ( 0 != $args['avatar_size'] ) {
							if ( empty( $comment_author_url ) ) {
								echo $avatar;
							} else {
								printf( '<a href="%s" rel="external nofollow" class="url">', $comment_author_url );
								echo $avatar;
							}
						}

						/*
						 * Using the `check` icon instead of `check_circle`, since we can't add a
						 * fill color to the inner check shape when in circle form.
						 */
						if ( radiance_is_comment_by_post_author( $comment ) ) {
							printf( '<span class="post-author-badge" aria-hidden="true">%s</span>', null );
						}

						printf(
							wp_kses(
								/* translators: %s: Comment author link. */
								__( ' %s <span class="screen-reader-text says">گفت:</span>', 'radiance' ),
								array(
									'span' => array(
										'class' => array(),
									),
								)
							),
							'<b class="fn">' . $comment_author . '</b>'
						);

						if ( ! empty( $comment_author_url ) ) {
							echo '</a>';
						}
						?>

                        <div class="comment-metadata pt-3">
                            <a href="<?php echo esc_url( get_comment_link( $comment, $args ) ); ?>">
                                <?php
                                /* translators: 1: Comment date, 2: Comment time. */
                                $comment_timestamp = sprintf( __( '%1$s at %2$s', 'radiance' ), get_comment_date( '', $comment ), get_comment_time() );
                                ?>
                                <time datetime="<?php comment_time( 'c' ); ?>" title="<?php echo $comment_timestamp; ?>">
                                    <?php echo $comment_timestamp; ?>
                                </time>
                            </a>
                            <?php
                            $edit_comment_icon = radiance_get_icon_svg( 'edit', 16 );
                            edit_comment_link( __( 'Edit', 'radiance' ), '<span class="edit-link-sep">&mdash;</span> <span class="edit-link">' . $edit_comment_icon, '</span>' );
                            ?>
                        </div><!-- .comment-metadata -->

					</div><!-- .comment-author -->

					<?php
					$commenter = wp_get_current_commenter();
					if ( $commenter['comment_author_email'] ) {
						$moderation_note = __( '<div class="alert alert-warning">نظر شما در انتظار تایید مدیر است!</div>', 'radiance' );
					} else {
						$moderation_note = __( '<div class="alert alert-warning py-0 px-2 my-2">این یک پیش نمایش از نظر شما می باشد. پس از تایید توسط مدیر سایت، نظر شما به کاربران نمایش داده می شود.</div>', 'radiance' );
					}
					?>

				</footer><!-- .comment-meta -->

				<div class="comment-content p-2">
					<?php comment_text(); ?>

                    <?php if ( '0' == $comment->comment_approved ) : ?>
                        <p class="comment-awaiting-moderation"><?php echo $moderation_note; ?></p>
                    <?php endif; ?>
				</div><!-- .comment-content -->

                <?php
                comment_reply_link(
                    array_merge(
                        $args,
                        array(
                            'add_below' => 'div-comment',
                            'depth'     => $depth,
                            'max_depth' => $args['max_depth'],
                            'before'    => '<div class="reply py-0 px-2 my-2"><span class="comment-reply comment-reply-link table-bordered p-1 rounded">',
                            'after'     => '</div></span>',
                            'reply_text'=> 'ارسال پاسخ'
                        )
                    )
                );
                ?>

			</article><!-- .comment-body -->
		<?php
	}
}
